#include <iostream>
#include <stdexcept>

// ---- Meta Assertions

#define STATIC_ASSERT(x) static const int STATIC_ASSERT_ ## __COUNT__ [(x)]

// ---- Runtime Error

struct out_of_range : virtual std::out_of_range {
  out_of_range() : std::out_of_range("value out of range") {}
};

// ---- Our Actual 'restricted' type

template<class T, T Min, T Max>
struct restricted {
  STATIC_ASSERT((Min < Max));
  
  T value;
  
  explicit restricted(T const& o)
    : value(o) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
  
  //
  // Just standard generic programming:
  // Writing a templated constructor.
  //
  // This allows to copy from restricted<> values with
  // different ranges.
  // 
  // Similar things are done for smart-ptr classes etc.
  //
  template<T OMin, T OMax>
  restricted(restricted<T, OMin, OMax> o)
    : value(o.value) {
    if(value < Min || Max < value) { throw out_of_range(); }
  }
};

// ---- How to use it

int main() {
  restricted<int, 0, 10> i(5);
  
  // copy from another restricted type:
  restricted<int, 0, 20> k(i);

  // btw, did we ever test the out-of-range handling?
  try {
    restricted<int, 0, 4> l(k);
    std::cerr << "error: out_of_range not thrown" << std::endl;
  } catch(out_of_range const&) {}

  // how about that?
  try {
    restricted<int, -10, -1> iAmToNegativeForMyLife(i);
    std::cerr << "error: out_of_range not thrown" << std::endl;
  } catch(out_of_range const&) {
    std::cerr
      << "!!!\n"
      << "!!! but hey: this should be a compile time check!\n"
      << "!!!" << std::endl;
  }
}
